<?php

use yii\helpers\Html;
use yii\grid\GridView;
use yii\bootstrap5\Alert;

/** @var yii\web\View $this */
/** @var yii\data\ActiveDataProvider $dataProvider */

$this->title = 'Мои заявки';
$this->params['breadcrumbs'][] = $this->title;
?>

<div class="application-index">
    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title"><?= Html::encode($this->title) ?></h1>
            <p class="page-subtitle">История переданных показаний и расчетов</p>
        </div>
        <?= Html::a('<i class="fas fa-plus"></i> Передать показания', ['create'], ['class' => 'btn-primary btn-add']) ?>
    </div>

    <div class="applications-container">
        <?= GridView::widget([
            'dataProvider' => $dataProvider,
            'tableOptions' => ['class' => 'applications-table'],
            'options' => ['class' => 'table-responsive'],
            'layout' => "{items}\n{pager}",
            'columns' => [
                [
                    'class' => 'yii\grid\SerialColumn',
                    'header' => '#',
                    'contentOptions' => ['class' => 'serial-column'],
                    'headerOptions' => ['class' => 'serial-header']
                ],
                [
                    'attribute' => 'date',
                    'format' => 'date',
                    'label' => 'Дата передачи',
                    'contentOptions' => ['class' => 'date-column'],
                    'headerOptions' => ['class' => 'date-header']
                ],
                [
                    'attribute' => 'category',
                    'value' => function ($model) {
                        return $model->getCategoryName();
                    },
                    'label' => 'Категория услуг',
                    'contentOptions' => ['class' => 'category-column'],
                    'headerOptions' => ['class' => 'category-header'],
                    'content' => function ($model) {
                        $icons = [
                            'electricity' => 'fas fa-bolt',
                            'water' => 'fas fa-tint',
                            'gas' => 'fas fa-fire'
                        ];
                        $icon = $icons[$model->category] ?? 'fas fa-file-alt';
                        return '<div class="category-cell"><i class="' . $icon . '"></i><span>' . $model->getCategoryName() . '</span></div>';
                    },
                    'format' => 'raw'
                ],
                [
                    'attribute' => 'previous_readings',
                    'value' => function ($model) {
                        return number_format($model->previous_readings, 2, ',', ' ');
                    },
                    'label' => 'Предыдущие',
                    'contentOptions' => ['class' => 'readings-column'],
                    'headerOptions' => ['class' => 'readings-header']
                ],
                [
                    'attribute' => 'current_readings',
                    'value' => function ($model) {
                        return number_format($model->current_readings, 2, ',', ' ');
                    },
                    'label' => 'Текущие',
                    'contentOptions' => ['class' => 'readings-column current'],
                    'headerOptions' => ['class' => 'readings-header']
                ],
                [
                    'attribute' => 'consumption',
                    'value' => function ($model) {
                        return number_format($model->consumption, 2, ',', ' ');
                    },
                    'label' => 'Расход',
                    'contentOptions' => ['class' => 'consumption-column'],
                    'headerOptions' => ['class' => 'consumption-header']
                ],
                [
                    'attribute' => 'tariff',
                    'value' => function ($model) {
                        return number_format($model->tariff, 2, ',', ' ') . ' руб.';
                    },
                    'label' => 'Тариф',
                    'contentOptions' => ['class' => 'tariff-column'],
                    'headerOptions' => ['class' => 'tariff-header']
                ],
                [
                    'attribute' => 'amount',
                    'value' => function ($model) {
                        return number_format($model->amount, 2, ',', ' ') . ' руб.';
                    },
                    'label' => 'Сумма',
                    'contentOptions' => ['class' => 'amount-column'],
                    'headerOptions' => ['class' => 'amount-header']
                ],
                [
                    'attribute' => 'status',
                    'value' => function ($model) {
                        return $model->getStatusName();
                    },
                    'format' => 'raw',
                    'label' => 'Статус',
                    'contentOptions' => ['class' => 'status-column'],
                    'headerOptions' => ['class' => 'status-header'],
                    'content' => function ($model) {
                        $statusClasses = [
                            'new' => 'status-new',
                            'processed' => 'status-processed',
                            'completed' => 'status-completed',
                            'rejected' => 'status-rejected'
                        ];
                        $class = $statusClasses[$model->status] ?? 'status-default';
                        return '<span class="status-badge ' . $class . '">' . $model->getStatusName() . '</span>';
                    }
                ],
                [
                    'attribute' => 'created_at',
                    'format' => 'datetime',
                    'label' => 'Создана',
                    'contentOptions' => ['class' => 'created-column'],
                    'headerOptions' => ['class' => 'created-header']
                ],
            ],
        ]); ?>
    </div>

    <?php if (!$dataProvider->getCount()): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <i class="fas fa-file-alt"></i>
            </div>
            <h3>Заявок пока нет</h3>
            <p>У вас еще нет переданных показаний. Создайте первую заявку!</p>
            <?= Html::a('Передать показания', ['create'], ['class' => 'btn-primary btn-empty']) ?>
        </div>
    <?php endif; ?>
</div>

<style>
    /* Page Header */
    .page-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-end;
        margin-bottom: 2rem;
        padding-bottom: 1.5rem;
        border-bottom: 1px solid #e8eeff;
    }

    .header-content {
        flex: 1;
    }

    .page-title {
        font-size: 2.25rem;
        font-weight: 700;
        color: #2c3e50;
        margin: 0 0 0.5rem 0;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .page-subtitle {
        color: #7f8c8d;
        font-size: 1.1rem;
        margin: 0;
    }

    /* Buttons */
    .btn-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        padding: 0.875rem 1.5rem;
        font-size: 1rem;
        font-weight: 600;
        border-radius: 12px;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
    }

    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        color: white;
        text-decoration: none;
    }

    .btn-add {
        white-space: nowrap;
    }

    .btn-empty {
        margin-top: 1rem;
    }

    /* Applications Container */
    .applications-container {
        background: white;
        border-radius: 16px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        overflow: hidden;
    }

    /* Table Styles */
    .applications-table {
        width: 100%;
        border-collapse: collapse;
        margin: 0;
    }

    .applications-table th {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        font-weight: 600;
        padding: 1.25rem 1rem;
        text-align: left;
        font-size: 0.9rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .applications-table td {
        padding: 1.25rem 1rem;
        border-bottom: 1px solid #f1f3f4;
        color: #2c3e50;
        font-size: 0.95rem;
    }

    .applications-table tr:last-child td {
        border-bottom: none;
    }

    .applications-table tr:hover {
        background: #f8faff;
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(102, 126, 234, 0.1);
        transition: all 0.2s ease;
    }

    /* Column Specific Styles */
    .serial-column {
        font-weight: 600;
        color: #667eea;
        text-align: center;
        width: 60px;
    }

    .date-column {
        font-weight: 500;
        white-space: nowrap;
    }

    .category-cell {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        font-weight: 500;
    }

    .category-cell i {
        color: #667eea;
        font-size: 1.1rem;
        width: 20px;
        text-align: center;
    }

    .readings-column {
        font-family: 'Courier New', monospace;
        font-weight: 600;
        text-align: right;
    }

    .readings-column.current {
        color: #27ae60;
        font-weight: 700;
    }

    .consumption-column {
        font-family: 'Courier New', monospace;
        font-weight: 600;
        text-align: right;
        color: #e67e22;
    }

    .tariff-column {
        font-family: 'Courier New', monospace;
        text-align: right;
        color: #7f8c8d;
    }

    .amount-column {
        font-family: 'Courier New', monospace;
        font-weight: 700;
        text-align: right;
        color: #2c3e50;
        font-size: 1rem;
    }

    .status-column {
        text-align: center;
    }

    .created-column {
        color: #7f8c8d;
        font-size: 0.9rem;
        white-space: nowrap;
    }

    /* Status Badges */
    .status-badge {
        padding: 0.5rem 1rem;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        display: inline-block;
    }

    .status-new {
        background: rgba(52, 152, 219, 0.1);
        color: #3498db;
        border: 1px solid rgba(52, 152, 219, 0.3);
    }

    .status-processed {
        background: rgba(243, 156, 18, 0.1);
        color: #f39c12;
        border: 1px solid rgba(243, 156, 18, 0.3);
    }

    .status-completed {
        background: rgba(46, 204, 113, 0.1);
        color: #27ae60;
        border: 1px solid rgba(46, 204, 113, 0.3);
    }

    .status-rejected {
        background: rgba(231, 76, 60, 0.1);
        color: #c0392b;
        border: 1px solid rgba(231, 76, 60, 0.3);
    }

    .status-default {
        background: rgba(149, 165, 166, 0.1);
        color: #7f8c8d;
        border: 1px solid rgba(149, 165, 166, 0.3);
    }

    /* Empty State */
    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        background: white;
        border-radius: 16px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .empty-icon {
        font-size: 4rem;
        color: #bdc3c7;
        margin-bottom: 1.5rem;
    }

    .empty-state h3 {
        color: #2c3e50;
        margin-bottom: 1rem;
        font-size: 1.5rem;
    }

    .empty-state p {
        color: #7f8c8d;
        font-size: 1.1rem;
        margin-bottom: 2rem;
    }

    /* Pagination */
    .pagination {
        display: flex;
        justify-content: center;
        padding: 2rem;
        gap: 0.5rem;
    }

    .pagination li {
        list-style: none;
    }

    .pagination li a {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 40px;
        height: 40px;
        border-radius: 8px;
        background: white;
        color: #667eea;
        text-decoration: none;
        font-weight: 600;
        border: 1px solid #e8eeff;
        transition: all 0.3s ease;
    }

    .pagination li.active a {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border-color: #667eea;
    }

    .pagination li a:hover {
        background: #667eea;
        color: white;
        transform: translateY(-1px);
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .applications-container {
            overflow-x: auto;
        }

        .applications-table {
            min-width: 1000px;
        }
    }

    @media (max-width: 768px) {
        .page-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 1rem;
        }

        .page-title {
            font-size: 1.75rem;
        }

        .applications-table th,
        .applications-table td {
            padding: 1rem 0.75rem;
            font-size: 0.85rem;
        }

        .empty-state {
            padding: 3rem 1rem;
        }

        .empty-icon {
            font-size: 3rem;
        }
    }

    /* Animation */
    @keyframes fadeIn {
        from {
            opacity: 0;
            transform: translateY(10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .applications-table tbody tr {
        animation: fadeIn 0.3s ease forwards;
    }

    .applications-table tbody tr:nth-child(1) { animation-delay: 0.1s; }
    .applications-table tbody tr:nth-child(2) { animation-delay: 0.2s; }
    .applications-table tbody tr:nth-child(3) { animation-delay: 0.3s; }
    .applications-table tbody tr:nth-child(4) { animation-delay: 0.4s; }
    .applications-table tbody tr:nth-child(5) { animation-delay: 0.5s; }

    /* Loading State */
    .applications-table.loading {
        opacity: 0.6;
        pointer-events: none;
    }

    .applications-table.loading::after {
        content: 'Загрузка...';
        display: block;
        text-align: center;
        padding: 2rem;
        color: #7f8c8d;
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Add hover effects to table rows
        const tableRows = document.querySelectorAll('.applications-table tbody tr');

        tableRows.forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
                this.style.boxShadow = '0 4px 12px rgba(102, 126, 234, 0.15)';
            });

            row.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
                this.style.boxShadow = 'none';
            });
        });

        // Add loading state for pagination
        const paginationLinks = document.querySelectorAll('.pagination a');

        paginationLinks.forEach(link => {
            link.addEventListener('click', function() {
                const table = document.querySelector('.applications-table');
                table.classList.add('loading');
            });
        });
    });
</script>